# SPDX-FileCopyrightText: StorPool <support@storpool.com>
# SPDX-License-Identifier: BSD-2-Clause
"""Test the parsing of base/URL pairs."""

from __future__ import annotations

from typing import NamedTuple

import pytest

from gifn_apply import repo_url


class TrivURL(NamedTuple):
    """The basic elements of a parsed URL."""

    scheme: str
    netloc: str
    path: str


PARSE_URLS: list[tuple[str, str, TrivURL | None]] = [
    ("", "https://github.com/openstack", None),
    ("ostack", "https://github.com/openstack", None),
    ("OSTACK!", "https://github.com/openstack", None),
    ("OSTACK", "https://github.com/openstack", TrivURL("https", "github.com", "/openstack/")),
    ("OSTACK", "https://github.com/openstack/", TrivURL("https", "github.com", "/openstack/")),
    ("local", "file:///absolute/path", None),
    ("LOCAL_3!", "file:///absolute/path", None),
    ("LOCAL_3", "file://host", None),
    ("LOCAL_3", "file:relative/path", None),
    ("LOCAL_3", "file:///absolute/path", TrivURL("file", "", "/absolute/path/")),
    ("OSTACK", "file:///absolute/path", TrivURL("file", "", "/absolute/path/")),
    ("OSTACK", "httpx://github.com/openstack", None),
    ("OSTACK", "loc", None),
    ("OSTACK", "/path", None),
    ("OSTACK", ":path", None),
]


@pytest.mark.parametrize(("base", "value", "expected"), PARSE_URLS)
def test_parse_url(base: str, value: str, expected: TrivURL | None) -> None:
    """Test the base parse_url() function."""
    res = repo_url.parse_url(f"R_{base}", base, value)
    if expected is None:
        assert isinstance(res, repo_url.RepoURLError)
    else:
        assert isinstance(res, repo_url.RepoURLOK)
        url = res.url.url
        assert (url.scheme, url.netloc, url.path) == expected
