# SPDX-FileCopyrightText: StorPool <support@storpool.com>
# SPDX-License-Identifier: BSD-2-Clause
"""Make sure the specified patches apply to the specified repositories."""

from __future__ import annotations

import argparse
import dataclasses
import functools
import logging
import pathlib
import sys
import tempfile
import typing

from . import defs
from . import gifn
from . import git
from . import quilt
from . import repo_url


if typing.TYPE_CHECKING:
    from typing import Final


@dataclasses.dataclass(frozen=True)
class RepoChanges:
    """A repo directory along with some metadata."""

    repo: defs.Repo
    repo_dir: pathlib.Path
    changes: list[str]
    changes_set: set[str]


@dataclasses.dataclass(frozen=True)
class Mode:
    """The abstract-ish base class for the gifn-apply tool's mode of operation."""


@dataclasses.dataclass(frozen=True)
class ModeHandled(Mode):
    """The parse_args() function handled everything, exit with the specified code."""

    return_code: int


@dataclasses.dataclass(frozen=True)
class ModeRun(Mode):
    """Run the tests with the specified config."""

    cfg: defs.Config


def _validate_series_filename(value: str) -> str:
    """Make sure `--series-file` specifies a non-empty filename, no path components."""
    path: Final = pathlib.Path(value)
    if len(path.parts) != 1:
        raise ValueError("The series filename may not contain any path components")
    if not path.parts[0]:
        raise ValueError("The series filename cannot be empty")
    return path.parts[0]


def _show_version() -> None:
    """Display program version information."""
    print(f"gifn-apply {defs.VERSION}")  # noqa: T201


def _show_features() -> None:
    """Display program features information."""
    print(f"Features: gifn-apply={defs.VERSION} repo-url=0.1 quilt=0.1")  # noqa: T201


def _build_logger(*, verbose: bool) -> logging.Logger:
    """Build a logger that outputs messages to the standard output stream."""
    logger: Final = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    stdout_handler: Final = logging.StreamHandler(stream=sys.stdout)
    stdout_handler.addFilter(lambda record: record.levelno == logging.INFO)
    stdout_handler.setLevel(logging.INFO)
    logger.addHandler(stdout_handler)

    stderr_handler: Final = logging.StreamHandler(stream=sys.stderr)
    if verbose:
        stderr_handler.addFilter(lambda record: record.levelno != logging.INFO)
        stderr_handler.setLevel(logging.DEBUG)
    else:
        stderr_handler.setLevel(logging.WARNING)
    logger.addHandler(stderr_handler)

    return logger


def _parse_args() -> Mode:
    """Parse the command-line arguments."""
    parser: Final = argparse.ArgumentParser(prog="gifn-apply")
    parser.add_argument(
        "--features",
        action="store_true",
        help="display information about supported program features and exit",
    )
    parser.add_argument(
        "-P",
        "--program",
        type=pathlib.Path,
        required=True,
        help="the path to the git-if-needed program to test",
    )
    parser.add_argument(
        "-p",
        "--patches",
        type=pathlib.Path,
        required=True,
        help="the path to the patch series file",
    )
    parser.add_argument(
        "-r",
        "--repo-url",
        type=repo_url.parse_base_url_pair,
        action="append",
        default=[],
        help="'base=url' pairs, e.g. 'openstack=https://github.com/openstack/'",
    )
    parser.add_argument(
        "-s",
        "--series-file",
        type=_validate_series_filename,
        default="series",
        help="the name of the series file in the patches directory (default: 'series')",
    )
    parser.add_argument(
        "-v",
        "--verbose",
        action="store_true",
        help="verbose operation; display diagnostic output",
    )
    parser.add_argument(
        "-V",
        "--version",
        action="store_true",
        help="display program version information and exit",
    )

    args: Final = parser.parse_args()

    if args.version:
        _show_version()
    if args.features:
        _show_features()
    if args.version or args.features:
        return ModeHandled(0)

    program: Final = args.program.resolve()
    if not program.is_file():
        sys.exit(f"Not a regular file: {program}")

    patches: Final = args.patches.resolve()
    if not patches.is_dir():
        sys.exit(f"Not a patches directory: {patches}")

    series_file: Final = args.series_file
    assert isinstance(series_file, str) and series_file  # noqa: S101,PT018

    repo_urls: Final = repo_url.get_env_repo_urls()
    for pair in args.repo_url:
        repo_urls[pair.base] = repo_url.RepoURLOK(pair.url)

    repo_urls_ok: Final = {}
    for base, res in sorted(repo_urls.items()):
        if isinstance(res, repo_url.RepoURLError):
            sys.exit(f"Invalid base URL for {base!r} / {res.name!r}: {res.value!r}: {res.err}")
        assert isinstance(res, repo_url.RepoURLOK)  # noqa: S101
        repo_urls_ok[base] = res.url

    return ModeRun(
        defs.Config(
            log=_build_logger(verbose=args.verbose),
            program=program,
            patches=patches,
            series=patches / series_file,
            repo_urls=repo_urls_ok,
        ),
    )


@functools.singledispatch
def _do_it(mode: Mode) -> None:
    """Do what the caller requested."""
    sys.exit(f"gifn-apply internal error: _do_it(): unhandled mode {mode!r}")


@_do_it.register
def _do_it_handled(mode: ModeHandled) -> None:
    """parse_args() did everything it wanted to, let's just go."""
    sys.exit(mode.return_code)


@_do_it.register
def _do_it_run(mode: ModeRun) -> None:
    """Run some tests."""
    cfg: Final = mode.cfg
    with tempfile.TemporaryDirectory(prefix="gifn-apply.") as tempd_obj:
        tempd: Final = pathlib.Path(tempd_obj)
        cfg.log.debug("Using %(tempd)s as a temporary directory", {"tempd": tempd})

        patches, repos_needed = quilt.parse_series(cfg)
        cfg.log.debug(
            "Need to check out %(repos)d repos, then apply %(patches)d patches",
            {"repos": len(repos_needed), "patches": len(patches)},
        )

        def list_change_ids(
            cfg: defs.Config,
            repo: defs.Repo,
            repo_dir: pathlib.Path,
        ) -> RepoChanges:
            """List the changes in the cloned or manipulated repository."""
            changes: Final = git.list_change_ids(cfg, repo_dir)
            return RepoChanges(repo, repo_dir, changes, set(changes))

        def clone_repo(repo: defs.Repo) -> RepoChanges:
            """Clone a single repo, get the list of changes."""
            cfg.log.info(
                "Cloning the %(repo)s repo from %(origin)s",
                {"repo": repo.repo, "origin": repo.origin},
            )
            repo_dir: Final = git.repo_clone(cfg, repo, tempd)
            res: Final = list_change_ids(cfg, repo, repo_dir)
            cfg.log.debug(
                "- cloned into %(repo_dir)s, got %(changes)d changes",
                {"repo_dir": res.repo_dir, "changes": len(res.changes)},
            )
            return res

        def check_repo(repo: defs.Repo, rchanges: RepoChanges) -> RepoChanges:
            """Make sure a repo contains all the changes applied before and maybe some more."""
            cfg.log.info(
                "Checking the %(repo)s repo from %(origin)s for new changes",
                {"repo": repo.repo, "origin": repo.origin},
            )
            if rchanges.repo != repo:
                sys.exit(
                    f"Internal error: {repo=!r} not the same as {rchanges.repo=!r} for "
                    f"{rchanges.repo_dir=!r} {len(rchanges.changes)} changes",
                )

            res: Final = list_change_ids(cfg, repo, rchanges.repo_dir)
            len_before: Final = len(rchanges.changes)
            len_after: Final = len(res.changes)
            if len_after < len_before:
                sys.exit(
                    f"gifn weirdness: {len_after=!r} < {len_before=!r} for "
                    f"{rchanges.repo=!r} at {rchanges.repo_dir=!r}",
                )
            if res.changes[:len_before] != rchanges.changes:
                sys.exit(
                    f"gifn weirdness: the first {len_before} changes are not the same for "
                    f"{rchanges.repo=!r} at {rchanges.repo_dir=!r}",
                )
            if not rchanges.changes_set.issubset(res.changes_set):
                sys.exit(
                    f"gifn weirdness: the {len_before=!r} changes are not "
                    f"contained within the {len_after=!r} ones for "
                    f"{rchanges.repo=!r} at {rchanges.repo_dir=!r}",
                )

            return res

        repos_before: Final = {repo: clone_repo(repo) for repo in repos_needed}

        cfg.log.info("Making sure that at least one of the changes has not been applied yet")
        unapplied_before_count: Final = len([
            patch
            for patch in patches
            if all(
                patch.change_id not in rchanges.changes_set for rchanges in repos_before.values()
            )
        ])
        if not unapplied_before_count:
            sys.exit("All the patches have been applied already")

        gifn.apply_series(cfg, tempd)
        repos_after: Final = {
            repo: check_repo(repo, rchanges) for repo, rchanges in repos_before.items()
        }

        cfg.log.info("Making sure that at least one of the changes has not been applied yet")
        unapplied_after: Final = [
            str(patch.relpath)
            for patch in patches
            if all(patch.change_id not in rchanges.changes_set for rchanges in repos_after.values())
        ]
        if unapplied_after:
            sys.exit(f"Some of the patches were not applied: {' '.join(unapplied_after)}")

        cfg.log.info("Everything seems to be fine!")


def main() -> None:
    """Parse command-line options, run the test."""
    _do_it(_parse_args())


if __name__ == "__main__":
    main()
