# vim: tabstop=4 shiftwidth=4 softtabstop=4

# Copyright 2012 OpenStack, LLC
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import json
import time
import urllib

from tempest.common.rest_client import RestClient
from tempest import exceptions


class ServersClientJSON(RestClient):

    def __init__(self, config, username, password, auth_url, tenant_name=None):
        super(ServersClientJSON, self).__init__(config, username, password,
                                                auth_url, tenant_name)
        self.service = self.config.compute.catalog_type

    def create_server(self, name, image_ref, flavor_ref, **kwargs):
        """
        Creates an instance of a server.
        name (Required): The name of the server.
        image_ref (Required): Reference to the image used to build the server.
        flavor_ref (Required): The flavor used to build the server.
        Following optional keyword arguments are accepted:
        adminPass: Sets the initial root password.
        key_name: Key name of keypair that was created earlier.
        meta: A dictionary of values to be used as metadata.
        personality: A list of dictionaries for files to be injected into
        the server.
        security_groups: A list of security group dicts.
        networks: A list of network dicts with UUID and fixed_ip.
        user_data: User data for instance.
        availability_zone: Availability zone in which to launch instance.
        accessIPv4: The IPv4 access address for the server.
        accessIPv6: The IPv6 access address for the server.
        min_count: Count of minimum number of instances to launch.
        max_count: Count of maximum number of instances to launch.
        disk_config: Determines if user or admin controls disk configuration.
        """
        post_body = {
            'name': name,
            'imageRef': image_ref,
            'flavorRef': flavor_ref
        }

        for option in ['personality', 'adminPass', 'key_name',
                       'security_groups', 'networks', 'user_data',
                       'availability_zone', 'accessIPv4', 'accessIPv6',
                       'min_count', 'max_count', ('metadata', 'meta'),
                       ('OS-DCF:diskConfig', 'disk_config')]:
            if isinstance(option, tuple):
                post_param = option[0]
                key = option[1]
            else:
                post_param = option
                key = option
            value = kwargs.get(key)
            if value is not None:
                post_body[post_param] = value
        post_body = json.dumps({'server': post_body})
        resp, body = self.post('servers', post_body, self.headers)

        body = json.loads(body)
        return resp, body['server']

    def update_server(self, server_id, name=None, meta=None, accessIPv4=None,
                      accessIPv6=None):
        """
        Updates the properties of an existing server.
        server_id: The id of an existing server.
        name: The name of the server.
        personality: A list of files to be injected into the server.
        accessIPv4: The IPv4 access address for the server.
        accessIPv6: The IPv6 access address for the server.
        """

        post_body = {}

        if meta is not None:
            post_body['metadata'] = meta

        if name is not None:
            post_body['name'] = name

        if accessIPv4 is not None:
            post_body['accessIPv4'] = accessIPv4

        if accessIPv6 is not None:
            post_body['accessIPv6'] = accessIPv6

        post_body = json.dumps({'server': post_body})
        resp, body = self.put("servers/%s" % str(server_id),
                              post_body, self.headers)
        body = json.loads(body)
        return resp, body['server']

    def get_server(self, server_id):
        """Returns the details of an existing server."""
        resp, body = self.get("servers/%s" % str(server_id))
        body = json.loads(body)
        return resp, body['server']

    def delete_server(self, server_id):
        """Deletes the given server."""
        return self.delete("servers/%s" % str(server_id))

    def list_servers(self, params=None):
        """Lists all servers for a user."""

        url = 'servers'
        if params:
            url += '?%s' % urllib.urlencode(params)

        resp, body = self.get(url)
        body = json.loads(body)
        return resp, body

    def list_servers_with_detail(self, params=None):
        """Lists all servers in detail for a user."""

        url = 'servers/detail'
        if params:
            url += '?%s' % urllib.urlencode(params)

        resp, body = self.get(url)
        body = json.loads(body)
        return resp, body

    def wait_for_server_status(self, server_id, status):
        """Waits for a server to reach a given status."""
        resp, body = self.get_server(server_id)
        server_status = body['status']
        start = int(time.time())

        while(server_status != status):
            time.sleep(self.build_interval)
            resp, body = self.get_server(server_id)
            server_status = body['status']

            if server_status == 'ERROR':
                raise exceptions.BuildErrorException(server_id=server_id)

            timed_out = int(time.time()) - start >= self.build_timeout

            if server_status != status and timed_out:
                message = ('Server %s failed to reach %s status within the '
                           'required time (%s s).' %
                           (server_id, status, self.build_timeout))
                message += ' Current status: %s.' % server_status
                raise exceptions.TimeoutException(message)

    def wait_for_server_termination(self, server_id, ignore_error=False):
        """Waits for server to reach termination."""
        start_time = int(time.time())
        while True:
            try:
                resp, body = self.get_server(server_id)
            except exceptions.NotFound:
                return

            server_status = body['status']
            if server_status == 'ERROR' and not ignore_error:
                raise exceptions.BuildErrorException(server_id=server_id)

            if int(time.time()) - start_time >= self.build_timeout:
                raise exceptions.TimeoutException

            time.sleep(self.build_interval)

    def list_addresses(self, server_id):
        """Lists all addresses for a server."""
        resp, body = self.get("servers/%s/ips" % str(server_id))
        body = json.loads(body)
        return resp, body['addresses']

    def list_addresses_by_network(self, server_id, network_id):
        """Lists all addresses of a specific network type for a server."""
        resp, body = self.get("servers/%s/ips/%s" %
                              (str(server_id), network_id))
        body = json.loads(body)
        return resp, body

    def change_password(self, server_id, password):
        """Changes the root password for the server."""
        post_body = {
            'changePassword': {
                'adminPass': password,
            }
        }

        post_body = json.dumps(post_body)
        return self.post('servers/%s/action' % str(server_id),
                         post_body, self.headers)

    def reboot(self, server_id, reboot_type):
        """Reboots a server."""
        post_body = {
            'reboot': {
                'type': reboot_type,
            }
        }

        post_body = json.dumps(post_body)
        return self.post('servers/%s/action' % str(server_id),
                         post_body, self.headers)

    def rebuild(self, server_id, image_ref, name=None, meta=None,
                personality=None, adminPass=None, disk_config=None):
        """Rebuilds a server with a new image."""
        post_body = {
                'imageRef': image_ref,
        }

        if name is not None:
            post_body['name'] = name

        if adminPass is not None:
            post_body['adminPass'] = adminPass

        if meta is not None:
            post_body['metadata'] = meta

        if personality is not None:
            post_body['personality'] = personality

        if disk_config is not None:
            post_body['OS-DCF:diskConfig'] = disk_config

        post_body = json.dumps({'rebuild': post_body})
        resp, body = self.post('servers/%s/action' % str(server_id),
                               post_body, self.headers)
        body = json.loads(body)
        return resp, body['server']

    def resize(self, server_id, flavor_ref, disk_config=None):
        """Changes the flavor of a server."""
        post_body = {
            'resize': {
                'flavorRef': flavor_ref,
            }
        }

        if disk_config is not None:
            post_body['resize']['OS-DCF:diskConfig'] = disk_config

        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % str(server_id),
                               post_body, self.headers)
        return resp, body

    def confirm_resize(self, server_id):
        """Confirms the flavor change for a server."""
        post_body = {
            'confirmResize': None,
        }

        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % str(server_id),
                               post_body, self.headers)
        return resp, body

    def revert_resize(self, server_id):
        """Reverts a server back to its original flavor."""
        post_body = {
            'revertResize': None,
        }

        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % str(server_id),
                               post_body, self.headers)
        return resp, body

    def create_image(self, server_id, image_name):
        """Creates an image of the given server."""
        post_body = {
            'createImage': {
                'name': image_name,
            }
        }

        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % str(server_id),
                               post_body, self.headers)
        return resp, body

    def list_server_metadata(self, server_id):
        resp, body = self.get("servers/%s/metadata" % str(server_id))
        body = json.loads(body)
        return resp, body['metadata']

    def set_server_metadata(self, server_id, meta):
        post_body = json.dumps({'metadata': meta})
        resp, body = self.put('servers/%s/metadata' % str(server_id),
                              post_body, self.headers)
        body = json.loads(body)
        return resp, body['metadata']

    def update_server_metadata(self, server_id, meta):
        post_body = json.dumps({'metadata': meta})
        resp, body = self.post('servers/%s/metadata' % str(server_id),
                               post_body, self.headers)
        body = json.loads(body)
        return resp, body['metadata']

    def get_server_metadata_item(self, server_id, key):
        resp, body = self.get("servers/%s/metadata/%s" % (str(server_id), key))
        body = json.loads(body)
        return resp, body['meta']

    def set_server_metadata_item(self, server_id, key, meta):
        post_body = json.dumps({'meta': meta})
        resp, body = self.put('servers/%s/metadata/%s' % (str(server_id), key),
                              post_body, self.headers)
        body = json.loads(body)
        return resp, body['meta']

    def delete_server_metadata_item(self, server_id, key):
        resp, body = self.delete("servers/%s/metadata/%s" %
                                 (str(server_id), key))
        return resp, body

    def stop(self, server_id):
        post_body = json.dumps({'os-stop': None})
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def start(self, server_id):
        post_body = json.dumps({'os-start': None})
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def attach_volume(self, server_id, volume_id, device='/dev/vdz'):
        """Attaches a volume to a server instance."""
        post_body = json.dumps({
            'volumeAttachment': {
                'volumeId': volume_id,
                'device': device,
            }
        })
        resp, body = self.post('servers/%s/os-volume_attachments' % server_id,
                               post_body, self.headers)
        return resp, body

    def detach_volume(self, server_id, volume_id):
        """Detaches a volume from a server instance."""
        resp, body = self.delete('servers/%s/os-volume_attachments/%s' %
                                 (server_id, volume_id))
        return resp, body

    def add_security_group(self, server_id, security_group_name):
        """Adds a security group to the server."""
        post_body = {
            'addSecurityGroup': {
                'name': security_group_name
            }
        }
        post_body = json.dumps(post_body)
        return self.post('servers/%s/action' % server_id,
                         post_body, self.headers)

    def remove_security_group(self, server_id, security_group_name):
        """Removes a security group from the server."""
        post_body = {
            'removeSecurityGroup': {
                'name': security_group_name
            }
        }
        post_body = json.dumps(post_body)
        return self.post('servers/%s/action' % server_id,
                         post_body, self.headers)

    def live_migrate_server(self, server_id, dest_host, use_block_migration):
        """This should be called with administrator privileges ."""

        migrate_params = {
            "disk_over_commit": False,
            "block_migration": use_block_migration,
            "host": dest_host
        }

        req_body = json.dumps({'os-migrateLive': migrate_params})

        resp, body = self.post("servers/%s/action" % str(server_id),
                               req_body, self.headers)
        return resp, body

    def list_servers_for_all_tenants(self):

        url = self.base_url + '/servers?all_tenants=1'
        resp = self.requests.get(url)
        resp, body = self.post('servers', post_body, self.headers)

        body = json.loads(body)
        return resp, body['servers']

    def migrate_server(self, server_id):
        """Migrates a server to a new host."""
        post_body = {'migrate': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)
        return resp, body

    def confirm_migration(self, server_id):
        """Confirms the migration of a server."""
        post_body = {'confirmResize': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)
        return resp, body

    def lock_server(self, server_id):
        """Locks the given server."""
        post_body = {'lock': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def unlock_server(self, server_id):
        """UNlocks the given server."""
        post_body = {'unlock': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def start_server(self, server_id):
        """Starts the given server."""
        post_body = {'os-start': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def stop_server(self, server_id):
        """Stops the given server."""
        post_body = {'os-stop': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def suspend_server(self, server_id):
        """Suspends the provded server."""
        post_body = {'suspend': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def resume_server(self, server_id):
        """Un-suspends the provded server."""
        post_body = {'resume': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def pause_server(self, server_id):
        """Pauses the provded server."""
        post_body = {'pause': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def unpause_server(self, server_id):
        """Un-pauses the provded server."""
        post_body = {'unpause': 'null'}
        post_body = json.dumps(post_body)
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)

    def reset_state(self, server_id, new_state='error'):
        """Resets the state of a server to active/error."""
        post_body = {
            'os-resetState': {
                'state': new_state
            }
        }
        resp, body = self.post('servers/%s/action' % server_id,
                               post_body, self.headers)
        return resp, body
